"""Represents the main game board

This is the level in which all the action takes place.
The board is a logical representation of the game board.

"""

import copy
from xml.etree.ElementTree import ElementTree
import networkx
import random

import serge.common
import serge.blocks.directions
import serge.blocks.layout

from theme import G
import squares
import enemy
import turret
import common

class OutOfRange(Exception): """The position was not in the board"""

FORWARD = 0
REVERSE = 1

TYPES = [
    {
        '1' : squares.Road(),
        '2' : squares.Wall(),
        '3' : squares.Surface(),
        '4' : squares.NearStart(),
        '5' : squares.Start(),
        '6' : squares.Goal(),
        '9' : squares.Ground(),
        '10' : squares.Goal(1),
        '11' : squares.Goal(2),
        '12' : squares.Goal(3),
    },
    {
        '1' : squares.Road(),
        '2' : squares.Wall(),
        '3' : squares.Surface(),
        '4' : squares.NearStart(),
        '5' : squares.Goal(1),
        '6' : squares.Goal(),
        '9' : squares.Ground(),
        '10' : squares.Start(),
        '11' : squares.Start(),
        '12' : squares.Start(),
    }
]

class GameBoard(serge.common.Loggable):
    """The board representing the game"""

    def __init__(self, direction=FORWARD):
        """Initialise the GameBoard"""
        self.addLogger()
        self._sprite_cache = {}
        self.dps_length_equivalence = G('dps-length-equivalence')
        self.item_types = TYPES[direction]
        
    @classmethod
    def fromXMLFile(cls, filename, direction):
        """Return a game board from an XML file"""
        #
        # Parse the file
        tree = ElementTree().parse(filename)
        layer = tree.find('layer')
        data = layer.find('data')
        #
        # Create the board
        ret = cls(direction)
        ret.data = []
        ret.starts = []
        ret.goals = []
        ret.turrets = []
        #
        for r, row in enumerate(data.text.strip().split('\n')):
            ret.data.append([])
            for c, col in enumerate(row.rstrip(',').split(',')):
                this_item = copy.copy(ret.item_types[col])
                ret.data[-1].append(this_item)
                if this_item.is_start:
                    ret.starts.append(this_item)
                if this_item.is_goal:
                    ret.goals.append(this_item)
                this_item.coords = (c, r)       
        #
        ret.makeTree()
        #
        return ret
        
    def getSize(self):
        """Return the size of the board"""
        return (len(self.data[0]), len(self.data))

    def getSpriteAt(self, (x, y)):
        """Return the sprite at a certain row and column location"""
        item = self.getItemAt((x, y))
        try:
            return self._sprite_cache[item.sprite]
        except KeyError:
            self.log.debug('Sprite cache miss for "%s". Creating now.' % item.sprite)
            sprite = serge.visual.Sprites.getItem(item.sprite)
            self._sprite_cache[item.sprite] = sprite
            return sprite

    def getItemAt(self, (x, y)):
        """Return the item at x and y"""
        try:
            return self.data[y][x]
        except IndexError:
            raise OutOfRange('The point (%f, %f) is not in the board' % (x, y))
            
    def makeTree(self):
        """Make the tree so that we can do pathfinding"""
        cols, rows = self.getSize()
        self.graph = networkx.Graph()
        for x in range(cols):
            for y in range(rows):
                item = self.getItemAt((x, y))
                if item.can_traverse and not item.is_frozen:
                    self.graph.add_node((x,y))
                    #
                    # Add edges
                    for direction in 'nesw':
                        dx, dy = serge.blocks.directions.getVectorFromCardinal(direction)
                        try:
                            new_item = self.getItemAt((x+dx, y+dy))
                        except OutOfRange:
                            pass
                        else:
                            if new_item.can_traverse and not new_item.is_frozen:
                                dps = self.getDamagePerSecondAtLocation((x, y))
                                weight = 1.0 + dps*self.dps_length_equivalence
                                full_weight = 1.0 + dps
                                self.graph.add_edge((x,y), (x+dx,y+dy), 
                                    weight=weight, full_weight=full_weight)
                            else:
                                self.graph.add_edge((x,y), (x+dx,y+dy), 
                                    weight=10000)
                                

    def getBestPath(self, start, end, ai):
        """Return the best path from the start to the end"""
        weight = 'weight' if ai == 'base' else 'full_weight'
        if start == end:
            return [end]
        try:
            return networkx.shortest_path(self.graph, start, end, weight=weight)
        except (networkx.exception.NetworkXNoPath, KeyError), err:
            raise common.NoPath('There is no path from %s to %s' % (start, end))
        except Exception, err:
            print 'Networkx error: %s' % err
            import pdb; pdb.set_trace()
        
    def getStartPosition(self):
        """Return a suitable starting position"""
        return random.choice(self.starts).coords
            
    def getGoalPosition(self, idx=None):
        """Return a suitable goal position"""
        goal_set = self.goals if not idx else [goal for goal in self.goals if goal.idx == idx]
        return random.choice(goal_set).coords

    def addTurret(self, (x, y), rnge, damage_per_second):
        """Add a turret to the board"""
        self.log.debug('Board added turret at (%s, %s) range %s, dps %s' % (x, y, rnge, damage_per_second))
        self.turrets.append(((x, y), rnge, damage_per_second))
        self.makeTree()
        
    def getDamagePerSecondAtLocation(self, (x, y)):
        """Return the expect damage per second at a certain location"""
        dps = 0
        for (tx, ty), rnge, damage in self.turrets:
            if (tx-x)**2 + (ty-y)**2 <= (rnge+2)**2:
                dps += damage
        return dps
            
    def getPossibleGoals(self):
        """Return the possible goal indexes"""
        return list(set([goal.idx for goal in self.goals]))
        
        
class BoardActor(serge.actor.CompositeActor):
    """An actor representing the board"""
    
    def __init__(self, tag, name, board, parent, (tw, th), background, show_tiles):
        """Initialise the actor"""
        super(BoardActor, self).__init__(tag, name)
        self.board = board
        self.parent = parent
        self.visual = serge.visual.SurfaceDrawing(G('screen-width'), G('screen-height'))
        self.occupied_squares = {}
        #
        # Create a layout to help position items
        self.tw, self.th = tw, th
        x, y = board.getSize()
        w, h = tw*x, th*y
        self.layout = serge.blocks.layout.BaseGrid('grid', 'grid', (x, y), w, h)
        #
        # Background if we have one
        self.bg = serge.visual.Sprites.getItem(background) if background else None
        self.show_tiles = show_tiles
        #
        self.createVisual()
        self.goal_index = 0

    def createVisual(self):
        """Create the visual for the board"""
        if self.bg:
            self.bg.renderTo(0, self.visual.getSurface(), (0, 0))
        if self.show_tiles:
            cols, rows = self.board.getSize()
            for row in range(rows):
                for col in range(cols):
                    sprite = self.board.getSpriteAt((col, row))
                    x, y = self.layout.getCoords((col, row))
                    cx, cy = x-self.tw/2, y-self.th/2
                    sprite.renderTo(0, self.visual.getSurface(), (cx, cy))

    def addEnemy(self, enemy_type):
        """Add an enemy"""
        e = enemy.Enemy('enemy', enemy_type, self.board, self.layout, self, self.goal_index)
        start = self.board.getStartPosition()
        x, y = self.layout.getCoords(start)
        e.moveTo(x, y)
        self.addChild(e)

    def updateActor(self, interval, world):
        """Update the actor"""
        super(BoardActor, self).updateActor(interval, world)
        #
        # Keep a running cache of which squares are occupied
        self.occupied_squares = {}
        for child in self.getChildren().findActorsByTag('enemy'):
            x, y = self.layout.getLocation((child.x, child.y))        
            self.occupied_squares.setdefault((x, y), set()).add(child)
        #
        # Unfreeze squares
        changed = False
        for child in self.getChildren().findActorsByTag('ice'):
            col, row = self.layout.getLocation((child.x, child.y))
            item = self.board.getItemAt((col, row))   
            if item.is_frozen:
                item.frozen_for -= interval/1000.0
                if item.frozen_for <= 0:
                    item.frozen_for = 0
                    item.is_frozen = False
                    changed = True
                    self.removeChild(child)
        #
        if changed:
            self.board.makeTree()            
            for target in self.getChildren().findActorsByTag('enemy'):
                target.clearPath()
            
    def canGoToSquare(self, (x, y), actor):
        """Return True if this actor can move to the given square"""
        occupiers = self.occupied_squares.get((x, y), set())
        if len(occupiers - set([actor])) == 0:
            return True
        else:
            return False

    def getOccupiersAt(self, (x, y)):
        """Return the occupiers of a certain square"""
        return self.occupied_squares.get((x, y), set())

    def canPlaceAt(self, name, (x, y)):
        """Return True if a turret could be placed at the x, y position"""
        dx, dy = G('size', name)
        for ix in range(x-dx/2, x+dx-dx/2+1):
            for iy in range(y-dy/2, y+dy-dy/2+1):
                try:
                    item = self.board.getItemAt((ix, iy))
                except OutOfRange:
                    return False
                if not item.can_place:
                    return False
        return True

    def placeItem(self, name, (x, y)):
        """Place an item on the board"""
        if G('type', name) == 'turret':
            item = self.placeTurret(name, (x, y))
        elif G('type', name) == 'ice':
            item = self.placeIce(name, (x, y))
        else:
            raise ValueError('Unknown item type for %s "%s"' % (name, G('type', name)))
        #
        # And alert all enemies that they should repath in case the new turret endangers them
        for target in self.getChildren().findActorsByTag('enemy'):
            target.clearPath()
        #
        return item
                    
    def placeTurret(self, name, (x, y)):
        """Place a turret at the x, y position - the new turret is returned"""
        self.log.info('Placing turret %s at %s' % (name, (x, y)))
        new_turret = turret.Turret(name, self, ((x, y)))
        self.addChild(new_turret)
        #
        # Block out any other turrets from being placed here
        dx, dy = G('size', name)
        cx, cy = self.layout.getLocation((x, y))
        for ix in range(cx-dx/2, cx+dx-dx/2+1):
            for iy in range(cy-dy/2, cy+dy-dy/2+1):
                try:
                    item = self.board.getItemAt((ix, iy))
                except OutOfRange:
                    pass
                else:
                    item.can_place = False
                    item.can_traverse = False
        #
        # Tell the board about the turret
        self.board.addTurret((cx+1, cy+1), G('range', name)/self.tw, G('damage', name)/G('speed', name))
        #
        return new_turret

    def placeIce(self, name, (x, y)):
        """Place ice on the board"""
        raise NotImplementedError('Need to update for exact x and y')
        self.log.info('Placing ice at %s, %s' % (x, y))
        cols, rows = self.board.getSize()
        cx, cy = self.layout.getCoords((x, y))
        for row in range(rows):
            for col in range(cols):
                item = self.board.getItemAt((col, row))   
                if item.can_freeze:
                    ix, iy = self.layout.getCoords((col, row))
                    if (ix-cx)**2 + (iy-cy)**2 <= G('range', name)**2:
                        item.is_frozen = True
                        item.frozen_for += G('freeze-strength', name)*(2.0-((ix-cx)**2 + (iy-cy)**2)/G('range', name)**2)/2.0
                        ice = serge.actor.Actor('ice', 'ice')
                        ice.setSpriteName('ice')
                        ice.setLayerName('ice')
                        ice.moveTo(ix, iy)
                        self.addChild(ice) 
        #
        # Tree needs to be recreated to account for the ice
        self.board.makeTree()
        #
        return None
        
    def getTargetsFor(self, (x, y), rnge):
        """Return a suitable target for a turret at the given location and range"""
        targets = []
        for target in self.getChildren().findActorsByTag('enemy'):
            offset = common.Vec2d(target.x, target.y) - common.Vec2d(x, y)
            if offset.length < rnge:
                targets.append((offset.length, target))
        return sorted(targets)
                
    def bulletLands(self, (x, y), bullet, impact_radius):
        """A bullet has landed"""
        for target in self.getChildren().findActorsByTag('enemy'):
            ex, ey = target.x, target.y
            offset = common.Vec2d(x, y) - common.Vec2d(ex, ey)
            if offset.length <= impact_radius:
                if target.takeDamage(bullet):
                    self.parent.destroyedEnemy(target)
        
    def enemyHitGoal(self, enemy):
        """An enemy hit the goal"""
        self.removeChild(enemy)
        self.parent.goalTakeDamage(enemy.damage)
        
    def setGoal(self, goal_index):
        """Switch enemies to a new goal"""
        self.log.info('Setting new goal index to %d' % goal_index)
        self.goal_index = goal_index
        self.getChildren().findActorsByTag('enemy').forEach().setGoalIndex(goal_index)
    
    def getGoal(self):
        """Return the current goal"""
        return self.goal_index
        
    def setSmartness(self, smartness):
        """Set the smartness of the AI in avoiding damage
        
        This goes from 100% smart to 0%. At 0% we don't avoid 
        damage at all. At 100% we try to avoid at all cost.
        
        """
        self.board.dps_length_equivalence = G('dps-length-equivalence')*smartness/100.0
        self.board.makeTree()
        self.getChildren().findActorsByTag('enemy').forEach().clearPath()
        
    def getPossibleGoals(self):
        """Return the possible goal indexes"""
        return self.board.getPossibleGoals()
